# Tiki file and directory organisation

## History and nature of this document

This path and file reoganisation exercise started in 2013 on [this thread](https://sourceforge.net/p/tikiwiki/mailman/tikiwiki-devel/thread/19C606DC-E4D5-4319-85D2-5DFE41F65363%40tiki.org/#msg31218770).  It was documented, then discussion continued on the wiki around 2017 [File and directory structure revamp](https://dev.tiki.org/File-and-directory-structure-revamp)  and [Path Structure](https://doc.tiki.org/Path%20Structure) but subsequently stalled.

It was apparently too far from the code to be kept up to date, and didn't separate clearly what is current documentation from future refactoring plans.

Today (2023) this document aims continue and finish the path reorganisation effort by being exhaustive enough to allow for a realistic progressive migration over time with no ambiguity.  It is ok if this takes years, as long as no developer has to wonder:

1. Where to put new stuff.
2. What the current paths do.
3. What the current plan is.

## Goals to achieve

* Developper onboarding and clarity
  * Allow developers to know what the content of a directory is about without having to review the whole code.
  * Keep reasonable directory depth, but minimize scrolling.
* Ease of sysadmin:
  * Minimise complexity of webserver configuration
  * Clarify and separate as much as possible:
   1. What needs and doesn't need to be backed-up.
   2. What needs to be served by the http server
   3. What needs to be writable from server side code.
* Modernization
  * Avoid versioning (committing to Git) generated files, and compatibility with modern build approaches like vite.js
  * Ease eventually moving all php code out of the server root (for increased security, etc.)

### Files at the root of tiki

Many need to stay here, either because:

* developers expect them there
* tooling expect them there and will pick them up by default (ex:  .gitignore, phpunit.xml.dist, .gitlab-ci.xml, etc.)
* They can really use the visibility even if they technically belong somewhere else (ex:  setup.sh, console.php)

* console.php Main tiki [console tool](https://doc.tiki.org/Console)

Most of the tiki-*.php will progressively move to either src/php/routes or public/

Some files a user would still have to backup individually if he uses/customizes them:

* .user.ini <https://www.php.net/manual/en/configuration.file.per-user.php>
* robot.txt (And there is a way to dynamically generate it via [tiki-robots.php](https://gitlab.com/tikiwiki/tiki/-/blob/master/tiki-robots.php))

## The directories

### TO KEEP (Currently implemented, or partially implemented)

This is the documentation for subdirectories, with hyperlinks to their respective README.md for more details.  If there is text here, its to clarify why a directory is NOT moving.  The README.md should be the place documenting what's in it.

* **[_custom/](_custom_dist/README.md)**
  * Custom code specific to a tiki installation.  Can be independently versioned, this directory is NOT present in git or in the tarball.
  * See _custom_dist for documentation
  * TODO: **lang_overrides/**
    * For what's currently in lang/??/custom.php (customFilePath from lib/language/LanguageTranslation.php)

* **[_custom_dist/](_custom_dist/README.md)**
  * Documentation and running code examples for _custom subsystem
  * Should be symlinked to _custom for developers working on improving the \_custom subsystem:

    ```bash
    ln -s _custom_dist _custom
    ```

* **bin/**  
  * This is already generated by php console.php dev:configure, but there is no placeholder or README.md in git.

* **[temp/](temp/README.md)**
  * Already exists, but progressively re-organise it and move more files to it.
  * Must not be backed-up, that's why it's not in private/.  Any file that cannot be reconstructed by tiki (such as mail queues) should go to private/storage
  * Shouldn't be expected to be readable over http
  * **[dev/](temp/dev/README.md)** For development/build/CI tools (vitejs, SCSS, GitLab-ci etc.).
  * **prod/** For tiki server operation, needs to be rw for PHP, and possibly other tools.
    * ^= storage/prefsdoc/
  * **export/**
    * ^= profiles/
      * Written to by console.php profile:export:init, and profile exports, which creates profiles/info.ini and profiles/name_of_exported_profile/
      * This is a write-only export process - benoitg, confirmed by jonny
* **[vendor/](vendor/README.md)**
* **[vendor_bundled/](vendor_bundled/README.md)**
* **[vendor_custom/](vendor_custom/README.md)**

### TO MOVE

#### legend

* **newpath/**
  * ^= oldpath (means content of oldpath is moved to newpath)
  * <= oldpath (means oldpath becomes a subdirectory of newpath: newpath/oldpath)

#### TODO

* **private/**
  * Eventually, all user files that should NOT be served over HTTP directly by the webserver will move here
  * Not versioned in Git (except example configurations and similar files)
  * The whole tree needs to be backed up
  * **config/**
    * ^= db/
    * ^= db/config
    * ^= db/tiki.ini.php (change the example system_configuration_file in ConfigureCommand.php)
  * **storage/**
    * Private storage.  Storage in the sense "Nonvolatile files manipulated by server code".  All files that need to be
      1. rw from PHP code
      2. do NOT need to be served over HTTP directly by the webserver
      3. Need to be backed-up
    * <= storage/fgal  According to the doc it should NOT be readable by webserver


* **public/**
  * All files that need to be served over HTTP directly by the webserver
  * ONLY storage/ should be writable by the web server user
  * Should be the (future) webserver root
    * This is still some way off, as for now:
      * vendor and vendor_bundled need to be accessible
  * **static/**
    * Images, fonts, etc. Sometimes called assets in other projects.  Versionned in Git.
    * <= img/
  * **generated/**
    * Any compiled or downloaded files to be served directly by the webserver.  Similar to the dist folder in pure JavaScript projects.  Empty in Git but filled in the tarballs.  Should not be backed up
  * **storage/**
    * Public storage.  Storage in the sense "Nonvolatile files manipulated by server code". All files that need to be
      1. rw from PHP code
      2. Need to be served over HTTP directly by the webserver
      3. Need to be backed-up (are not volatile)
    * ^= storage/*  but only files that need to be served by the web server directly. So NOT fgal, prefsdoc and tiki-manager.
    * <= img/trackers/ as img_tracker_field (As of 2023-04-13, this is still used by lib/core/Tracker/Field/Image.php)
    * <= img/wiki_up/ as img_plugin_img (As of 2023-04-13, this is still used by lib/wikiplugin/wikiplugin_img.php)
    * =< whelp/ as structure_webhelp (As of 2023-04-13, this is still used by lib/structures/structlib.php)
  * **temp/**  
    * Only temp files that need to be readable over http and do NOT need to be backed-up.
    * ^= [temp/public](temp/public/README.md)

* **tests/** (both Vue and phpunit use testS, plural, so we do to)
  * **js/**
    * Only if we have js without the tests alongside, which will likely not be the case
  * **php/**
    * Alternative would be phpunit/, if we are not going to have the tests beside the code, may as well split them by tooling.
    * <= [lib/test/](lib/test/README.md)* split out, mostly in unit/ Naming mostly from <https://docs.phpunit.de/en/10.0/>
    * **unit/** Most current tests
* **themes/**
  * Themes have conflicting requirements
    1. They need some protection, as themes contain code (in the form of tpl files)
    2. They need to be self-contained to be useful.  As they contain assets, it implied they (or at least part of them) need to be http readable by every user.
    3. The theme installer needs to be able to write them to maintain functionality.
    4. Installed themes (or user custom ones) need to survive upgrades.
    5. They need to be distributed with their bundled files (specifically scss => css), or php needs to be able to dynamically generated them when they are installed.
  * This requires some compromise, php needs to be able to write them, the web server needs to be able to serve parts of them
  * This directory will be split in 4 and removed.  It will make versioning easier, and things clearer:
    1. public/static/base_themes for themes distributed in tiki.  This require the theme lookup code to look in two places.
    2. public/storage/themes for themes installed from the theme installer or user custom themes.
    3. public/storage/custom_css and public/storage/custom_js for behaviours currently in themes/css/custom.css and themes/js/custom.js.  The naming is still up for discussion.
    4. private/template_overrides/ for behaviours currently in themes/templates/

* **tools/**
  * Developer tools and scripts. The difference with bin is that these cannot be called directly
    * ^= [doc/devtools](README.md)
    * <= [permissioncheck/](permissioncheck/README.md)

* **src/**
  * <= [installer/](installer/README.md)
  * **js/**
    * Should be organised according to framework and build systems.
    * Formerly static js should only move as they are processed by the build system, ideally we should avoid moving them to public/static/js
    * <= lib/vue-mf/
    * <= lib/jquery_tiki/ (lib/jquery_tiki/elfinder can move to to lib/elfinder, it has php files)
    * <= lib/tiki-js.js
    * <= lib/ckeditor_tiki
  * **lang/**
    * ^= [lang/](lang/README.md)
  * **lib/**
    * This needs to happen progressively, as this directory needs to be ASSUMED not part of the document root (although that is not yet the case). So the js files need to move first.
    * Internal reoganisations is out of scope until the sub-directories all have a README.md
    * ^= Most of current [lib/](lib/README.md)
    * <= [admin/](admin/README.md), renamed as adminpages/
    * **routes/**
      * <= Most of the current tiki-*.php if/when we move to a php routing solution
      * <= lists
  * **templates/**
    * ^= [templates/](templates/README.md)
    * They mirror the structure of the php sources, but are technically smarty, and are overridable in themes, and other mechanisms, and thus don't go into php/
  * **modules/**
    * ^= [modules/](modules/README.md)
  * **wikiplugins**
    * ^= [lib/wikiplugins/](lib/wikiplugins/README.md)

#### UNKNOWN paths
These paths may or may not still be used by code, or need more information to determine where they should move to.

* **doc/reports/**
  * What does that do, what uses it?

### To DELETE paths (including content)
As far as we can tell, these are not referenced by any active code and are ready to be removed from git, setup scripts, .gitignore, etc.

Some of these may be created by scripts, and not currently versioned in git.

* **tests/**
  * Was that commited in error?

## Generated files

Some generated files will always be in git (ex:  package manager lock files like composer.lock), other will remain for some time.  

* **vendor_bundled/composer.lock**
  * Generated from vendor_bundled/composer.json using ``composer -d vendor_bundled update``
* **package.lock.json**
  * Generated from */package.json using ``npm install``
* **doc/devtools/codesniffer/standards/TikiIgnore/generate_ignore_list.php**
  * Generated using `php doc/devtools/codesniffer/standards/TikiIgnore/generate_ignore_list.php`.

It is useful to list theme here, so developers that hit merge conflicts can regenerate them instead of spending hours resolving conflicts manually.

A typical flow to resolve merge conflicts in one of these would be (rebase example):

```bash
git rebase upstream/master
git checkout --theirs vendor_bundled/composer.lock
git add vendor_bundled/composer.lock
git rebase --continue
// Regenerate the file
composer -d vendor_bundled update
git add vendor_bundled/composer.lock
git commit --amend
```

## Discussion/justifications

### How does the code even know about this?

ASIDE from the features that actually use these paths, we have many partial lists of these paths in string format.  As of 2023-05-17, that is at least:

* installer/installlib.php in create_dirs()
* tiki-check.php, look for the $dirs[] variable.
* installlib.php, look for the $dirs[] variable.
* lib/core/Tiki/Command/MultiTikiMoveCommand.php, look for $dirs
* setup.sh, look for DIRS=

We need to unify these in proper constants, and end up with 2 files to update, one for PHP, one for bash.

### 2 aspects were not considered in previous discussions

1. The importance of JavaScript kept rising, which now (2023) makes not requiring npm impossible for development environments.
2. The local/ and system/ separation that was planned is not realistic at top level, because it would preclude us from ever moving anything out of the document root, which makes security and ease of sysadmin harder (and a separate document root is the way PHP development is going).

### Removing generated files from Git, and depending on node for javascript build

There are currently many generated files (or trivially generatable files) in git:

* index.php and .htaccess designed to disallow directory listing in apache.  We already have code in php doc/devtools/check_tiki_directories.php to check and even generate (php doc/devtools/check_tiki_directories.php fix-index).  And another, different one, in lib/core/Tiki/Composer/CleanVendors.php (addIndexFiles())
* Vue3 javascript files currently in /lib/vue_mf
* A limited number of other js files.
* scss files
  * In themes, they need to be dynamically recompiled.  Currently this is done in php and is extremely slow.  Node can do it much faster, with more actively maintained processor.  Since we will require functional node for git install, this would only be a problem for tarball installs.  But since people using tarball installs usually have to have a local tiki to work on their themes, and they then upload it anyway, noting is lost, as long as themes are distributed with their css pre-compiled.
  * In the rest of the codebase this will just improve the developer experience and allow a watch process with no downside.

In any event, tarball installs over ftp with no build process at all should still be supported.

### Moving the document root in the future

Most files can be moved to public as shown above.  But there are remaining problems:

Currently the webserver serves files directly from themes, and from vendor_bundled.  Unfortunately:

* themes/
  * must be readable by webserver, or proxyed because they contain assets.  Moving them to public is probably acceptable (but not ideal), as the only code they contain is .tpl.
* vendor_bundled/
  1. Some composer modules we use may not work outside the document root.  But that must be few of them these days
  2. A lot of what's there are javascript modules from <https://asset-packagist.org/>.  
  * I really don't think asset-packagist is a realistic (or even useful) long term solution.  We can't tree-shake packages, and they don't fit well in a modern javascript development flow, which is enough of a moving target as it is.
  * Most of them will naturally move to a npm/vite build process as time goes on.
* vendor_custom/ and vendor/
  1. These can contain Tiki packages <https://doc.tiki.org/Packages>, which include themes, profiles, css, etc.
  1. The good news is that they already copy themes to themes/ in pure php, we'd have to do the same for custom js and css (that's js and css to support the php code, so it can't simply be sent to themes).  

Drupal has the same requirement as we currently do of being installed in the document root <https://www.drupal.org/project/drupal/issues/1672986>.  This is what the Symfony Asset component is designed to manage, but it would be a lot of refactoring for tiki.

While we will now require npm for git installs, we can't rely on that for ftp installs.

### The problem with shared hosting without shell access

My proposal is the following:

1. Stop supporting "real" tiki at the document root
2. Most users should be able to set the DocumentRoot somewhere on the web hosting.
3. For those that can't (ex: cPanel primary domains), provide a .htaccess at the tiki root that will rewrite requests transparently for Apache.
4. Is there really anybody not covered by the above?

.htaccess at root would be something like

    RewriteEngine on
    RewriteCond %{REQUEST_URI} !^/public
    RewriteRule ^(.*)$ /public/$1 [NC,L]`
