<?php

/**
 * Copyright 2014 Fabian Grutschus. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * The views and conclusions contained in the software and documentation are those
 * of the authors and should not be interpreted as representing official policies,
 * either expressed or implied, of the copyright holders.
 *
 * @author    Fabian Grutschus <f.grutschus@lubyte.de>
 * @copyright 2014 Fabian Grutschus. All rights reserved.
 * @license   BSD
 * @link      http://github.com/fabiang/xmpp
 */

namespace Fabiang\Xmpp\Event;

use PHPUnit\Framework\TestCase;

/**
 * Generated by PHPUnit_SkeletonGenerator 1.2.1 on 2013-12-31 at 19:16:20.
 *
 * @coversDefaultClass Fabiang\Xmpp\Event\EventManager
 */
class EventManagerTest extends TestCase
{

    /**
     * @var EventManager
     */
    protected $object;

    /**
     * Sets up the fixture, for example, opens a network connection.
     * This method is called before a test is executed.
     *
     * @return void
     */
    protected function setUp(): void
    {
        $this->object = new EventManager;
    }

    /**
     * Tears down the fixture, for example, closes a network connection.
     * This method is called after a test is executed.
     *
     * @return void
     */
    protected function tearDown(): void
    {

    }

    /**
     * Test attaching events.
     *
     * @covers ::attach
     * @covers ::getEventList
     * @uses Fabiang\Xmpp\Event\EventManager::__construct
     * @uses Fabiang\Xmpp\Event\EventManager::setEventObject
     * @return void
     */
    public function testAttach()
    {
        $this->object->attach('foo', function() {});
        $this->assertEquals(
            array('*' => array(), 'foo' => array(function() {})),
            $this->object->getEventList()
        );
        $this->object->attach('foo', function() {});
        $this->assertEquals(
            array('*' => array(), 'foo' => array(function() {}, function() {})),
            $this->object->getEventList()
        );
    }

    /**
     * Test attaching callback only once.
     *
     * @covers ::attach
     * @covers ::getEventList
     * @uses Fabiang\Xmpp\Event\EventManager::__construct
     * @uses Fabiang\Xmpp\Event\EventManager::setEventObject
     * @return void
     */
    public function testAttachDublicates()
    {
        $callback = function() {

        };
        $this->object->attach('foo', $callback);
        $this->assertSame(array('*' => array(), 'foo' => array($callback)), $this->object->getEventList());
        $this->object->attach('foo', $callback);
        $this->assertSame(array('*' => array(), 'foo' => array($callback)), $this->object->getEventList());
    }

    /**
     * Test that happens when no valid callback is passwd to attach method.
     *
     * @covers ::attach
     * @uses Fabiang\Xmpp\Event\EventManager::__construct
     * @uses Fabiang\Xmpp\Event\EventManager::setEventObject
     * @expectedException \Fabiang\Xmpp\Exception\InvalidArgumentException
     * @return void
     */
    public function testAttachInvalidCallback()
    {
        $this->object->attach('foo', array());
    }

    /**
     * Test triggering events.
     *
     * @covers ::trigger
     * @uses Fabiang\Xmpp\Event\EventManager::__construct
     * @uses Fabiang\Xmpp\Event\EventManager::attach
     * @uses Fabiang\Xmpp\Event\EventManager::setEventObject
     * @uses Fabiang\Xmpp\Event\EventManager::getEventObject
     * @uses Fabiang\Xmpp\Event\Event
     * @return void
     */
    public function testTrigger()
    {
        $eventObject      = null;
        $previousListener = function() {};

        $this->object->attach('foo', $previousListener);
        $this->object->attach('foo', function ($event) use(&$eventObject) {
            $eventObject = $event;
        });
        $this->object->trigger('foo', new \stdClass, array(1, 2, 3));

        $this->assertSame('foo', $eventObject->getName());
        $this->assertInstanceof('stdClass', $eventObject->getTarget());
        $this->assertSame(array(1, 2, 3), $eventObject->getParameters());
        $this->assertSame(array($previousListener), $eventObject->getEventStack());
    }

    /**
     * Test triggering wildcard listeners.
     *
     * @covers ::trigger
     * @uses Fabiang\Xmpp\Event\EventManager::__construct
     * @uses Fabiang\Xmpp\Event\EventManager::attach
     * @uses Fabiang\Xmpp\Event\EventManager::setEventObject
     * @uses Fabiang\Xmpp\Event\EventManager::getEventObject
     * @uses Fabiang\Xmpp\Event\Event
     * @return void
     */
    public function testTriggerWildcard()
    {
        $eventObject = null;
        $this->object->attach('*', function ($event) use(&$eventObject) {
            $eventObject = $event;
        });
        $this->object->trigger('foo', new \stdClass, array(1, 2, 3));
        $this->assertNotNull($eventObject);
    }

    /**
     * Test triggering events.
     *
     * @covers ::trigger
     * @uses Fabiang\Xmpp\Event\EventManager::__construct
     * @uses Fabiang\Xmpp\Event\EventManager::attach
     * @uses Fabiang\Xmpp\Event\EventManager::setEventObject
     * @return void
     */
    public function testTriggerNoCallbacksAttached()
    {
        $eventObject = null;
        $this->object->attach('foo', function ($event) use(&$eventObject) {
            $eventObject = $event;
        });
        $this->object->trigger('bar', new \stdClass, array(1, 2, 3));

        $this->assertNull($eventObject);
    }

    /**
     * Test setting and getting event object.
     *
     * @covers ::getEventObject
     * @covers ::setEventObject
     * @uses Fabiang\Xmpp\Event\EventManager::__construct
     * @return void
     */
    public function testSetAndGetEventObject()
    {
        $eventObject = new Event;
        $this->assertSame($eventObject, $this->object->setEventObject($eventObject)->getEventObject());
    }

    /**
     * Test constructor.
     *
     * @covers ::__construct
     * @uses Fabiang\Xmpp\Event\EventManager::getEventObject
     * @uses Fabiang\Xmpp\Event\EventManager::setEventObject
     * @return void
     */
    public function testConstructor()
    {
        $object = new EventManager;
        $this->assertInstanceOf('\\Fabiang\\Xmpp\\Event\\Event', $object->getEventObject());

        $eventObject = new Event;
        $object      = new EventManager($eventObject);
        $this->assertSame($eventObject, $object->getEventObject());
    }

}
